//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for QuickTime Music Architecture sample.
//				This file is used for BOTH MacOS and Windows.
//
//	Written by:	Tim Monroe
//				Based (heavily!) on the MovieShell code written by Apple DTS.
//
//	Copyright:	 1994-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <13>	 	02/03/99	rtm		removed "\p" escape sequences from strings
//	   <12>	 	12/16/98	rtm		removed all QTVR API calls, so we can remove QTVR.lib from project
//	   <11>	 	04/07/98	rtm		begun adding text support
//	   <10>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <9>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu identifiers
//	   <8>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <7>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <6>	 	06/04/97	rtm		removed call to QTVRUtils_IsQTVRMovie in InitApplicationWindowObject
//	   <5>	 	02/06/97	rtm		fixed window resizing code
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//	   
//////////

// header files
#include "ComApplication.h"
#include "QTText.h"

// global variables for Macintosh code
#if TARGET_OS_MAC
Str255					gAppName = "\pQTText";					// the name of this application
#endif

// global variables for Windows code
#if TARGET_OS_WIN32
extern HWND				ghWnd;									// the MDI frame window; this window has the menu bar
extern int				gNumWindowsOpen;
extern LPSTR			gCmdLine;
#endif

long					gMaxMilliSecToUse = 0L;		

extern Boolean			gSearchForward;
extern Boolean			gSearchWrap;
extern Boolean			gSearchWithCase;
extern Str255			gSearchText;
extern Str255			gSampleText;


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// *before* the MDI frame window is created or *after*. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// ***do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {
		QTText_CopyCStringToPascal(kSearchText, gSearchText);
		QTText_CopyCStringToPascal(kSampleText, gSampleText);
	}	// end of kInitAppPhase_BeforeCreateFrameWindow

	// ***do any start-up activities that should occur after the MDI frame window is created
	if (theStartPhase & kInitAppPhase_AfterCreateFrameWindow) {
		
#if TARGET_OS_WIN32
		// on Windows, open as movie documents any files specified on the command line
		DoOpenCommandLineMovies(gCmdLine);									
#endif

	}	// end of kInitAppPhase_AfterCreateFrameWindow
}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// *before* any open movie windows are destroyed or *after*.
//
//////////

void StopApplication (UInt32 theStopPhase)
{

}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif	// TARGET_OS_MAC
		}
	}

	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	BeginUpdate(GetPortFromWindowReference(theWindow));
	
	// draw the movie controller and its movie
	MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, theWindow);
	
	EndUpdate(GetPortFromWindowReference(theWindow));
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// @@@INSERT APPLICATION-SPECIFIC CONTENT CLICKING FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean		isHandled = true;
	
	switch (theCharCode) {
	
		// @@@HANDLE APPLICATION-SPECIFIC KEY PRESSES HERE

		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
	WindowObject		myWindowObject = NULL;
	ApplicationDataHdl	myAppData = NULL;
	MovieController 	myMC = NULL;
	Movie			 	myMovie = NULL;
	OSErr				myErr = noErr;
	
	myWindowObject = GetWindowObjectFromFrontWindow();
	if (myWindowObject != NULL) {
		myMC = (**myWindowObject).fController;
		myMovie = (**myWindowObject).fMovie;
		myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
	}
	
	switch (theMenuItem) {
	
		case IDM_SET_TEXT:
			// put up a dialog box to get a text string
			QTText_SetSearchText();
			break;
				
		case IDM_FIND_TEXT:
			QTText_FindText(myWindowObject, gSearchText);
			break;
				
		case IDM_EDIT_TEXT:
			QTText_EditText(myWindowObject);
			break;
				
		case IDM_SEARCH_FORWARD:
			gSearchForward = true;	
			break;
				
		case IDM_SEARCH_BACKWARD:
			gSearchForward = false;	
			break;
				
		case IDM_WRAP_SEARCH:
			gSearchWrap = !gSearchWrap;	
			break;
				
		case IDM_USE_CASE:
			gSearchWithCase = !gSearchWithCase;	
			break;
				
		case IDM_ADD_TEXT_TRACK:
			{
				// add a text track to the specified movie;
				// for purposes of illustration, we'll add 3 (count 'em!) text strings to the movie,
				// with each string occupying about one-third of the movie duration
				
				Track		myTypeTrack = NULL;
				Track		myTextTrack = NULL;
				TimeValue	myMovieDuration = 0;
				TimeValue	mySampleDuration = 0;
				char		*myStrings[] = {"Text 1", "Text 2", "Text 3"};
				short		myFrames[3];
				Boolean		isChapter = true;
				
				myTypeTrack = GetMovieIndTrackType(myMovie, 1, VideoMediaType, movieTrackMediaType);
				if (myTypeTrack == NULL)
					break;
					
				// get the duration of the movie and the duration of a single frame;
				// this tells us how many frames fit into one-third of the movie
				myMovieDuration = GetMovieDuration(myMovie);
				mySampleDuration = QTUtils_GetFrameDuration(myTypeTrack);
				
				myFrames[0] = (myMovieDuration / mySampleDuration) / 3;
				myFrames[1] = (myMovieDuration / mySampleDuration) / 3;
				myFrames[2] = (myMovieDuration / mySampleDuration) - (myFrames[0] + myFrames[1]);				
								
				myTextTrack = QTText_AddTextTrack(myMovie, myStrings, myFrames, 3, VideoMediaType, isChapter);
				if (myTextTrack != NULL) {

					QTText_UpdateMovieAndController(myMC);

					// stamp the movie as dirty and update our saved data
					(**myWindowObject).fDirty = true;
					(**myAppData).fMovieHasText = true;
					(**myAppData).fTextIsChapter = isChapter;
					(**myAppData).fTextTrack = myTextTrack;
					(**myAppData).fTextHandler = GetMediaHandler(GetTrackMedia(myTextTrack));
				}
			}
			break;
				
		case IDM_CUT_TEXT_TRACK:
			// remove all existing text tracks from the specified movie
			myErr = QTText_RemoveIndTextTrack(myMovie, kAllTextTracks);
			if (myErr == noErr) {
			
				QTText_UpdateMovieAndController(myMC);

				// stamp the movie as dirty and update our saved data
				(**myWindowObject).fDirty = true;
				(**myAppData).fMovieHasText = false;
				(**myAppData).fTextIsChapter = false;
				(**myAppData).fTextTrack = NULL;
				(**myAppData).fTextHandler = NULL;
			}
			break;
				
		case IDM_CHAPTER_TRACK:
			(**myAppData).fTextIsChapter = !(**myAppData).fTextIsChapter;
			QTText_SetTextTrackAsChapterTrack(myWindowObject, VideoMediaType, (**myAppData).fTextIsChapter);
			break;
			
		default:
			break;
	} // switch (theMenuItem)
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
	WindowObject		myWindowObject = NULL; 
	ApplicationDataHdl	myAppData = NULL;
	MovieController 	myMC = NULL;
	MenuReference		myMenu;
	
#if TARGET_OS_WIN32
	myMenu = theMenu;
#elif TARGET_OS_MAC
	myMenu = GetMenuHandle(kTestMenu);
#endif
	
	if (theWindow != NULL)
		myWindowObject = GetWindowObjectFromWindow(theWindow);
	
	// assume it's all disabled
	SetMenuItemState(myMenu, IDM_SET_TEXT, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_FIND_TEXT, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_EDIT_TEXT, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_SEARCH_FORWARD, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_SEARCH_BACKWARD, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_WRAP_SEARCH, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_USE_CASE, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_ADD_TEXT_TRACK, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_CUT_TEXT_TRACK, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_CHAPTER_TRACK, kDisableMenuItem);

	// set check marks
	SetMenuItemCheck(myMenu, IDM_SEARCH_FORWARD, gSearchForward);
	SetMenuItemCheck(myMenu, IDM_SEARCH_BACKWARD, !gSearchForward);
	SetMenuItemCheck(myMenu, IDM_WRAP_SEARCH, gSearchWrap);
	SetMenuItemCheck(myMenu, IDM_USE_CASE, gSearchWithCase);
	
	if (myWindowObject != NULL) {
		myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
		if ((myAppData != NULL) && ((**myAppData).fMovieHasText)) {
			SetMenuItemState(myMenu, IDM_SET_TEXT, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_FIND_TEXT, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_EDIT_TEXT, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_SEARCH_FORWARD, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_SEARCH_BACKWARD, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_WRAP_SEARCH, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_USE_CASE, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_CHAPTER_TRACK, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_CUT_TEXT_TRACK, kEnableMenuItem);
			SetMenuItemCheck(myMenu, IDM_CHAPTER_TRACK, (**myAppData).fTextIsChapter);
		} else {
			SetMenuItemState(myMenu, IDM_ADD_TEXT_TRACK, kEnableMenuItem);
		}
	}
	
	// we don't allow creating new files here...
#if TARGET_OS_MAC
	SetMenuItemState(GetMenuHandle(mFile), iNew, kDisableMenuItem);
#endif
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
	return(false);			// no-op for now
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{		
	if (theWindowObject != NULL)
		(**theWindowObject).fAppData = (Handle)QTText_InitWindowData(theWindowObject);
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
	if (theWindowObject != NULL)
		QTText_DumpWindowData(theWindowObject);
	
	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	switch (theAction) {
	
		// handle window resizing
		case mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			break;

		// handle idle events
		case mcActionIdle:
			DoIdle((**myWindowObject).fWindow);
			break;
			
		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}


